import { FuzzyLevel, FuzzyRule, Priority, SensorName } from './fuzzy.types';
import { FLV } from './FLV';

/**
 * FuzzyEngine is a class that implements a fuzzy logic engine.
 * It takes fuzzy logic variables (FLVs) and rules to evaluate input data
 * and produce a score and label based on the fuzzy logic rules.
 */
export class FuzzyEngine {
  constructor(
    private flvs: Partial<Record<SensorName, FLV>>,
    private rules: FuzzyRule[],
  ) {}

  /**
   * Evaluates the input data using the fuzzy logic rules and variables.
   * @param input - The input data to evaluate.
   * @returns An object containing the score and label based on the fuzzy logic evaluation.
   */
  evaluate(input: Partial<Record<SensorName, number>>): {
    score: number;
    label: Priority;
  } {
    const fuzzified = this.fuzzifyInputs(input);
    const scores = { Low: 0, Medium: 0, High: 0, VeryHigh: 0 };

    for (const rule of this.rules) {
      let strength = 1;
      for (const [sensor, level] of Object.entries(rule.conditions)) {
        const memberships = fuzzified[sensor as SensorName];
        if (!memberships) {
          strength = 0;
          break;
        }
        strength *= memberships[level];
      }
      scores[rule.output] += strength;
    }

    const crisp =
      (scores.Low * 25 +
        scores.Medium * 50 +
        scores.High * 75 +
        scores.VeryHigh * 100) /
      (scores.Low + scores.Medium + scores.High + scores.VeryHigh || 1);

    const label: Priority =
      crisp > 85
        ? 'VeryHigh'
        : crisp > 65
          ? 'High'
          : crisp > 40
            ? 'Medium'
            : 'Low';

    return { score: parseFloat(crisp.toFixed(2)), label };
  }

  /**
   * Fuzzifies the input data based on the fuzzy logic variables (FLVs).
   * @param input - The input data to fuzzify.
   * @returns An object containing the fuzzified values for each sensor.
   */
  private fuzzifyInputs(
    input: Partial<Record<SensorName, number>>,
  ): Partial<Record<SensorName, Record<FuzzyLevel, number>>> {
    const result: Partial<Record<SensorName, Record<FuzzyLevel, number>>> = {};
    for (const [sensor, value] of Object.entries(input)) {
      const flv = this.flvs[sensor as SensorName];
      if (flv) {
        result[sensor as SensorName] = flv.GetDoM(value);
      }
    }
    return result;
  }
}
